<?php

namespace App\Services;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class MekariTalentaSandboxClient
{
    private string $baseUrl;
    private string $clientId;
    private string $clientSecret;

    public function __construct()
    {
        $this->baseUrl      = (string) config('services.mekari_talenta_sandbox.base_url');
        $this->clientId     = (string) config('services.mekari_talenta_sandbox.username');
        $this->clientSecret = (string) config('services.mekari_talenta_sandbox.secret');
    }

    private function buildPathWithQuery(string $path, array $query): string
    {
        if (empty($query)) return $path;
        $qs = http_build_query($query, '', '&', PHP_QUERY_RFC3986);
        return $path . (str_contains($path, '?') ? '&' : '?') . $qs;
    }

    private function sign(string $method, string $pathWithQuery, ?array $body = null): array
    {
        $date = gmdate('D, d M Y H:i:s \G\M\T');
        $requestLine  = strtoupper($method) . " {$pathWithQuery} HTTP/1.1";
        $stringToSign = "date: {$date}\n{$requestLine}";
        $signature    = base64_encode(hash_hmac('sha256', $stringToSign, $this->clientSecret, true));

        $headers = [
            'Authorization' => sprintf(
                'hmac username="%s", algorithm="hmac-sha256", headers="date request-line", signature="%s"',
                $this->clientId,
                $signature
            ),
            'Date'          => $date,
            'Accept'        => 'application/json',
        ];

        if (in_array(strtoupper($method), ['POST','PUT'], true)) {
            $json = json_encode($body ?? []);
            $headers['Digest'] = 'SHA-256=' . base64_encode(hash('sha256', $json ?: '', true));
            $headers['Content-Type'] = 'application/json';
        }

        Log::debug('[SANDBOX SIGN]', ['method'=>strtoupper($method), 'reqLine'=>$requestLine]);
        return $headers;
    }

    private function requestFullPath(string $method, string $path, array $query = [], ?array $body = null): array
    {
        $pathWithQuery = $this->buildPathWithQuery($path, $query);
        $headers       = $this->sign($method, $pathWithQuery, $body);
        $http = Http::baseUrl($this->baseUrl)->withHeaders($headers);

        $resp = match (strtoupper($method)) {
            'GET'    => $http->get($pathWithQuery),
            'POST'   => $http->post($pathWithQuery, $body ?? []),
            'PUT'    => $http->put($pathWithQuery, $body ?? []),
            'DELETE' => $http->delete($pathWithQuery, $body ?? []),
            default  => throw new \InvalidArgumentException("Unsupported method {$method}"),
        };

        $finalUrl = $this->baseUrl . $pathWithQuery;
        if ($resp->failed()) {
            Log::error('[SANDBOX ERROR]', ['status'=>$resp->status(),'body'=>$resp->body(),'url'=>$finalUrl]);
            throw new \RuntimeException("Talenta Sandbox error: " . $resp->body());
        }

        Log::debug('[SANDBOX OK]', ['status'=>$resp->status(),'url'=>$finalUrl]);
        return (array) $resp->json();
    }

    public function get(string $path, array $query = []): array
    { return $this->requestFullPath('GET', $path, $query); }

    public function post(string $path, array $body = [], array $query = []): array
    { return $this->requestFullPath('POST', $path, $query, $body); }

    public function put(string $path, array $body = [], array $query = []): array
    { return $this->requestFullPath('PUT', $path, $query, $body); }

    public function delete(string $path, array $body = [], array $query = []): array
    { return $this->requestFullPath('DELETE', $path, $query, $body); }
}
